"use strict";
// Copyright Amazon.com Inc. or its affiliates. All Rights Reserved.
// SPDX-License-Identifier: Apache-2.0
Object.defineProperty(exports, "__esModule", { value: true });
exports.increment = exports.rawDerive = exports.kdfCounterMode = exports.INT32_MAX_LIMIT = void 0;
/*
 * Implementation of the https://nvlpubs.nist.gov/nistpubs/SpecialPublications/NIST.SP.800-108r1.pdf
 * Key Derivation in Counter Mode Using Pseudorandom Functions. This
 * implementation mirrors the Dafny one: https://github.com/aws/aws-cryptographic-material-providers-library/blob/main/AwsCryptographyPrimitives/src/KDF/KdfCtr.dfy
 */
const crypto_1 = require("crypto");
const material_management_1 = require("@aws-crypto/material-management");
const serialize_1 = require("@aws-crypto/serialize");
const SEPARATION_INDICATOR = Buffer.from([0x00]);
const COUNTER_START_VALUE = 1;
exports.INT32_MAX_LIMIT = 2147483647;
const SUPPORTED_IKM_LENGTHS = [32, 48, 66];
const SUPPORTED_NONCE_LENGTHS = [16, 32];
const SUPPORTED_DERIVED_KEY_LENGTHS = [32, 64];
const SUPPORTED_DIGEST_ALGORITHMS = ['sha256', 'sha384'];
function kdfCounterMode({ digestAlgorithm, ikm, nonce, purpose, expectedLength, }) {
    /* Precondition: the ikm must be 32, 48, 66 bytes long */
    (0, material_management_1.needs)(SUPPORTED_IKM_LENGTHS.includes(ikm.length), `Unsupported IKM length ${ikm.length}`);
    /* Precondition: the nonce is required */
    (0, material_management_1.needs)(nonce, 'The nonce must be provided');
    /* Precondition: the nonce must be 16, 32 bytes long */
    (0, material_management_1.needs)(SUPPORTED_NONCE_LENGTHS.includes(nonce.length), `Unsupported nonce length ${nonce.length}`);
    /* Precondition: the expected length must be 32, 64 bytes */
    /* Precondition: the expected length * 8 must be under the max 32-bit signed integer */
    (0, material_management_1.needs)(SUPPORTED_DERIVED_KEY_LENGTHS.includes(expectedLength) &&
        expectedLength * 8 < exports.INT32_MAX_LIMIT &&
        expectedLength * 8 > 0, `Unsupported requested length ${expectedLength}`);
    const label = purpose || Buffer.alloc(0);
    const info = nonce;
    const internalLength = 8 + SEPARATION_INDICATOR.length;
    /* Precondition: the input length must be under the max 32-bit signed integer */
    (0, material_management_1.needs)(internalLength + label.length + info.length < exports.INT32_MAX_LIMIT, `Input Length ${internalLength + label.length + info.length} must be under ${exports.INT32_MAX_LIMIT} bytes`);
    const lengthBits = Buffer.from((0, serialize_1.uInt32BE)(expectedLength * 8));
    const explicitInfo = Buffer.concat([
        label,
        SEPARATION_INDICATOR,
        info,
        lengthBits,
    ]);
    return rawDerive(ikm, explicitInfo, expectedLength, digestAlgorithm);
}
exports.kdfCounterMode = kdfCounterMode;
function rawDerive(ikm, explicitInfo, length, 
// omit offset as a parameter because it is unused, causing compile errors due
// to configured project settings
digestAlgorithm) {
    const h = (0, crypto_1.createHash)(digestAlgorithm).digest().length;
    /* Precondition: expected length must be positive */
    (0, material_management_1.needs)(length > 0, `Requested length ${length} must be positive`);
    /* Precondition: length of explicit info + 4 bytes should be under the max 32-bit signed integer */
    (0, material_management_1.needs)(4 + explicitInfo.length < exports.INT32_MAX_LIMIT, `Explicit info length ${explicitInfo.length} must be under ${exports.INT32_MAX_LIMIT - 4} bytes`);
    /* Precondition: the digest algorithm should be sha256 */
    (0, material_management_1.needs)(SUPPORTED_DIGEST_ALGORITHMS.includes(digestAlgorithm), `Unsupported digest algorithm ${digestAlgorithm}`);
    /* Precondition: the expected length + digest hash length should be under the max 32-bit signed integer - 1 */
    (0, material_management_1.needs)(length + h < exports.INT32_MAX_LIMIT - 1, `The combined requested and digest hash length ${length + h} must be under ${exports.INT32_MAX_LIMIT - 1} bytes`);
    // number of iterations calculated in accordance with SP800-108
    const iterations = Math.floor((length + h - 1) / h);
    let buffer = Buffer.alloc(0);
    let i = Buffer.from((0, serialize_1.uInt32BE)(COUNTER_START_VALUE));
    for (let iteration = 1; iteration <= iterations; iteration++) {
        const digest = (0, crypto_1.createHmac)(digestAlgorithm, ikm)
            .update(i)
            .update(explicitInfo)
            .digest();
        buffer = Buffer.concat([buffer, digest]);
        i = increment(i);
    }
    (0, material_management_1.needs)(buffer.length >= length, 'Failed to derive key of requested length');
    return buffer.subarray(0, length);
}
exports.rawDerive = rawDerive;
function increment(x) {
    /* Precondition: buffer length must be 4 bytes */
    (0, material_management_1.needs)(x.length === 4, `Buffer length ${x.length} must be 4 bytes`);
    let output;
    if (x[3] < 255) {
        output = Buffer.from([x[0], x[1], x[2], x[3] + 1]);
    }
    else if (x[2] < 255) {
        output = Buffer.from([x[0], x[1], x[2] + 1, 0]);
    }
    else if (x[1] < 255) {
        output = Buffer.from([x[0], x[1] + 1, 0, 0]);
    }
    else if (x[0] < 255) {
        output = Buffer.from([x[0] + 1, 0, 0, 0]);
    }
    else {
        throw new Error('Unable to derive key material; may have exceeded limit.');
    }
    /* Postcondition: incremented buffer length must be 4 bytes */
    (0, material_management_1.needs)(output.length === 4, `Incremented buffer length ${output.length} must be 4 bytes`);
    return output;
}
exports.increment = increment;
//# sourceMappingURL=data:application/json;base64,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